@extends('layouts.invoices')

@section('content')

<div class="container">
    <!-- Tab Navigation -->
    <ul class="nav nav-tabs mb-3" id="invoiceTabs">
        <li class="nav-item">
            <a class="nav-link active" id="invoices-tab" data-bs-toggle="pill" 
               data-bs-target="#invoices-pane" href="#" role="tab" 
               aria-controls="invoices-pane" aria-selected="true">
                <i class="fas fa-file-invoice me-2"></i> Existing Invoices
            </a>
        </li>
        <li class="nav-item">
            <a class="nav-link" id="voucher-tab" data-bs-toggle="pill" 
               data-bs-target="#voucher-pane" href="#" role="tab" 
               aria-controls="voucher-pane" aria-selected="false">
                <i class="fas fa-receipt me-2"></i> Payment Vouchers
            </a>
        </li>
        <li class="nav-item">
            <a class="nav-link" id="expenses-tab" data-bs-toggle="pill" 
               data-bs-target="#expenses-pane" href="#" role="tab" 
               aria-controls="expenses-pane" aria-selected="false">
                <i class="fas fa-chart-line me-2"></i> Monthly Expenses
            </a>
        </li>
    </ul>

    <!-- Tab Content -->
    <div class="tab-content" id="invoiceTabsContent">
       <!-- Existing Invoices Tab -->
<div class="tab-pane fade show active" id="invoices-pane" role="tabpanel" aria-labelledby="invoices-tab" tabindex="0">
    <!-- Search + Filters Card -->
    <div class="card shadow-sm border-light rounded mb-4">
        <div class="card-header bg-info text-white">
            <div class="d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Filter Invoices</h5>
                <button type="button" class="btn btn-light" data-bs-toggle="modal" data-bs-target="#createInvoiceModal">
                    <i class="fas fa-plus me-2"></i>Create New Invoice
                </button>
            </div>
        </div>
        <div class="card-body">
            <form method="GET" action="{{ route('invoices.index') }}">
                <div class="row g-2 align-items-end">
                    <!-- Invoice Number -->
                    <div class="col-md">
                        <label class="form-label">Invoice Number</label>
                        <div class="input-group">
                            <span class="input-group-text"><i class="fas fa-file-invoice"></i></span>
                            <input type="text" name="invoice_number" class="form-control" 
                                   placeholder="Invoice Number" value="{{ request('invoice_number') }}">
                        </div>
                    </div>

                    <!-- Agency -->
                    <div class="col-md">
                        <label class="form-label">Agency</label>
                        <div class="input-group">
                            <span class="input-group-text"><i class="fas fa-building"></i></span>
                            <select name="external_agency_id" class="form-select">
                                <option value="">All Agencies</option>
                                @foreach($agencies as $agency)
                                    <option value="{{ $agency->id }}" 
                                        {{ request('external_agency_id') == $agency->id ? 'selected' : '' }}>
                                        {{ $agency->company_name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                    </div>

                    <!-- Status -->
                    <div class="col-md">
                        <label class="form-label">Status</label>
                        <div class="input-group">
                            <span class="input-group-text"><i class="fas fa-check-circle"></i></span>
                            <select name="status" class="form-select">
                                <option value="">All Statuses</option>
                                @foreach(['draft', 'pending', 'paid', 'overdue', 'cancelled'] as $status)
                                    <option value="{{ $status }}" 
                                        {{ request('status') == $status ? 'selected' : '' }}>
                                        {{ ucfirst($status) }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                    </div>

                    <!-- Date Range -->
                    <div class="col-md">
                        <label class="form-label">From Date</label>
                        <input type="date" name="from_date" class="form-control" 
                               value="{{ request('from_date') }}">
                    </div>

                    <div class="col-md">
                        <label class="form-label">To Date</label>
                        <input type="date" name="to_date" class="form-control" 
                               value="{{ request('to_date') }}">
                    </div>

                    <!-- Submit & Clear -->
                    <div class="col-auto">
                        <button type="submit" class="btn btn-info mt-2">
                            <i class="fas fa-filter"></i> Filter
                        </button>
                        <a href="{{ route('invoices.index') }}" class="btn btn-outline-secondary mt-2 ms-2">
                            <i class="fas fa-times"></i> Clear
                        </a>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Invoices List Card -->
    <div class="card shadow-sm border-light rounded">
        <div class="card-header bg-light">
            <h5 class="mb-0">Invoice List</h5>
        </div>
        <div class="card-body">
            <div class="row">
                @foreach($invoices as $invoice)
                    @php
                        $workersData = is_array($invoice->workers_data) ? $invoice->workers_data : json_decode($invoice->workers_data, true);
                        $workerCount = is_array($workersData) ? count($workersData) : 0;
                    @endphp
                    <div class="col-12 mb-4">
                        <div class="card shadow rounded">
                            <div class="card-body">
                                <div class="row align-items-center">
                                    <!-- Invoice Number + Date -->
                                    <div class="col-md-3">
                                        <div class="text-muted small">Invoice #</div>
                                        <strong>{{ $invoice->invoice_number }}</strong>
                                        <div class="text-muted small mt-1">Date</div>
                                        <div>{{ $invoice->invoice_date->format('Y-m-d') }}</div>
                                    </div>

                                    <!-- Agency Info -->
                                    <div class="col-md-2">
                                        <div class="text-muted small">e_Agency</div>
                                        <strong>{{ $invoice->client_name }}</strong>
                                    </div>

                                    <!-- Workers Count -->
                                    <div class="col-md-1 text-center">
                                        <div class="text-muted small">Workers</div>
                                        <span class="badge bg-primary">{{ $workerCount }}</span>
                                    </div>

                                    <!-- Financial Info -->
                                    <div class="col-md-3">
                                        <div class="row">
                                            <div class="col-4">
                                                <div class="text-muted small">Stamping</div>
                                                <div>{{ number_format($invoice->stamping_fee_total, 2) }}</div>
                                            </div>
                                            <div class="col-4">
                                                <div class="text-muted small">Arrival</div>
                                                <div>{{ number_format($invoice->arrival_fee_total, 2) }}</div>
                                            </div>
                                            <div class="col-4">
                                                <div class="text-muted small">Total</div>
                                                <strong>{{ number_format($invoice->total_amount, 2) }}</strong>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Status -->
                                    <div class="col-md-1">
                                        <div class="text-muted small">Status</div>
                                        <span class="badge bg-{{ $invoice->status == 'paid' ? 'success' : 'info' }}">
                                            {{ ucfirst($invoice->status) }}
                                        </span>
                                    </div>

                                    <!-- Actions -->
                                    <div class="col-md-2 text-end">
                                        <div class="d-flex gap-2 justify-content-end">
                                            <a href="{{ route('invoices.show', $invoice->id) }}" class="btn btn-sm btn-info" title="View">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                            <a href="{{ route('invoices.edit', $invoice->id) }}" class="btn btn-sm btn-primary" title="Edit">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                            @if($invoice->status != 'paid')
                                            <form action="{{ route('invoices.markAsPaid', $invoice->id) }}" method="POST" class="d-inline">
                                                @csrf
                                                <button type="submit" class="btn btn-sm btn-success" title="Mark as Paid">
                                                    <i class="fas fa-check"></i>
                                                </button>
                                            </form>
                                            @endif
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                @endforeach

                <!-- No Results Message -->
                <div id="noInvoicesMessage" class="text-center text-muted w-100 my-4" 
                     style="{{ count($invoices) ? 'display: none;' : '' }}">
                    <p>No invoices found matching your criteria.</p>
                </div>
            </div>

            <!-- Pagination -->
            <div class="d-flex justify-content-center mt-3">
                {{ $invoices->links() }}
            </div>
        </div>
    </div>
</div>

        <!-- Payment Voucher Tab -->
        <div class="tab-pane fade" id="voucher-pane" role="tabpanel" aria-labelledby="voucher-tab">
            <!-- Quick Actions Bar -->
            <div class="card border-0 shadow-sm mb-4">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div class="d-flex gap-2">
                            <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createVoucherModal">
                                <i class="fas fa-plus me-2"></i>Create New Voucher
                            </button>
                            <button type="button" class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#addSubAgentModal">
                                <i class="fas fa-user-plus me-2"></i>Add Sub-Agent
                            </button>
                        </div>
                        <div class="d-flex gap-2">
                            <div class="input-group" style="width: 300px;">
                                <span class="input-group-text bg-light">
                                    <i class="fas fa-search"></i>
                                </span>
                                <input type="text" class="form-control" id="voucherSearch" placeholder="Search vouchers...">
                            </div>
                            <select class="form-select" id="voucherStatusFilter" style="width: 150px;">
                                <option value="">All Status</option>
                                <option value="pending">Pending</option>
                                <option value="approved">Approved</option>
                                <option value="rejected">Rejected</option>
                            </select>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Vouchers List -->
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-light">
                    <div class="d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Payment Vouchers</h5>
                        <div class="d-flex gap-2">
                            <button class="btn btn-outline-secondary btn-sm" id="exportVouchers">
                                <i class="fas fa-file-export me-1"></i>Export
                            </button>
                            <button class="btn btn-outline-secondary btn-sm" id="printVouchers">
                                <i class="fas fa-print me-1"></i>Print
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    @if($vouchers->isEmpty())
                        <div class="text-center py-5">
                            <i class="fas fa-receipt fa-3x text-muted mb-3"></i>
                            <h5 class="text-muted">No vouchers found</h5>
                            <p class="text-muted">Create your first payment voucher using the button above</p>
                        </div>
                    @else
                        <div class="table-responsive">
                            <table class="table table-hover align-middle">
                                <thead>
                                    <tr>
                                        <th>
                                            <input type="checkbox" class="form-check-input" id="selectAllVouchers">
                                        </th>
                                        <th>Voucher #</th>
                                        <th>Sub-Agent</th>
                                        <th>Type</th>
                                        <th>Workers</th>
                                        <th>Amount</th>
                                        <th>Status</th>
                                        <th>Created</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($vouchers as $voucher)
                                        <tr>
                                            <td>
                                                <input type="checkbox" class="form-check-input voucher-checkbox" 
                                                       value="{{ $voucher->id }}">
                                            </td>
                                            <td>
                                                <span class="fw-bold">{{ $voucher->voucher_number }}</span>
                                            </td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="avatar-sm me-2">
                                                        <span class="avatar-title rounded-circle bg-primary">
                                                            {{ substr($voucher->subAgent->name, 0, 1) }}
                                                        </span>
                                                    </div>
                                                    <div>
                                                        <div class="fw-medium">{{ $voucher->subAgent->name }}</div>
                                                        <small class="text-muted">{{ $voucher->subAgent->email }}</small>
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <span class="badge bg-{{ $voucher->voucher_type === 'advance' ? 'info' : 'success' }}">
                                                    {{ ucfirst($voucher->voucher_type) }}
                                                </span>
                                            </td>
                                            <td>
                                                <span class="badge bg-secondary">
                                                    @php
                                                        $workersData = $voucher->workers_data ? 
                                                            (is_array($voucher->workers_data) ? 
                                                                $voucher->workers_data : 
                                                                json_decode($voucher->workers_data, true)
                                                            ) : [];
                                                        $workerCount = is_array($workersData) ? count($workersData) : 0;
                                                    @endphp
                                                    {{ $workerCount }} {{ Str::plural('Worker', $workerCount) }}
                                                </span>
                                            </td>
                                            <td>
                                                <div class="fw-bold">{{ number_format($voucher->total_amount, 2) }}</div>
                                                <small class="text-muted">{{ $voucher->currency }}</small>
                                            </td>
                                            <td>
                                                <span class="badge bg-{{ $voucher->status === 'pending' ? 'warning' : 
                                                    ($voucher->status === 'approved' ? 'success' : 'danger') }}">
                                                    {{ ucfirst($voucher->status) }}
                                                </span>
                                            </td>
                                            <td>
                                                <div>{{ $voucher->created_at->format('M d, Y') }}</div>
                                                <small class="text-muted">{{ $voucher->created_at->format('h:i A') }}</small>
                                            </td>
                                            <td>
                                                <div class="btn-group">
                                                    <button type="button" class="btn btn-sm btn-info" 
                                                            onclick="viewVoucher({{ $voucher->id }})" title="View Details">
                                                        <i class="fas fa-eye"></i>
                                                    </button>
                                                    @if($voucher->status === 'pending')
                                                        <button type="button" class="btn btn-sm btn-success" 
                                                                onclick="approveVoucher({{ $voucher->id }})" title="Approve">
                                                            <i class="fas fa-check"></i>
                                                        </button>
                                                        <button type="button" class="btn btn-sm btn-danger" 
                                                                onclick="rejectVoucher({{ $voucher->id }})" title="Reject">
                                                            <i class="fas fa-times"></i>
                                                        </button>
                                                    @endif
                                                    <button type="button" class="btn btn-sm btn-secondary" 
                                                            onclick="exportVoucher({{ $voucher->id }})" title="Export PDF">
                                                        <i class="fas fa-file-pdf"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>

                        <!-- Pagination -->
                        <div class="d-flex justify-content-between align-items-center mt-4">
                            <div class="text-muted">
                                Showing {{ $vouchers->firstItem() }} to {{ $vouchers->lastItem() }} 
                                of {{ $vouchers->total() }} entries
                            </div>
                            <div>
                                {{ $vouchers->links() }}
                            </div>
                        </div>
                    @endif
                </div>
            </div>
        </div>


@include('invoices.partials.monthly-expenses-tab')


        <!-- Office Expenses Tab -->
        <div class="tab-pane fade" id="office-expenses-pane" role="tabpanel" aria-labelledby="office-expenses-tab" tabindex="0">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <h5 class="card-title">Office Expenses</h5>
                    <p class="card-text">Office expenses content will go here.</p>
                </div>
            </div>
        </div>
    </div>
</div>
@include('invoices.partials.subagents-modal')

<!-- Create Voucher Modal -->
<div class="modal fade" id="createVoucherModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Create Payment Voucher</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="voucherForm" action="{{ route('vouchers.store') }}" method="POST">
                    @csrf
                    <div class="row g-3">
                        <!-- Sub-Agent Selection -->
                        <div class="col-md-6">
                            <label class="form-label required">Sub-Agent</label>
                            <select class="form-select" name="sub_agent_id" id="subAgentSelect" required>
                                <option value="">Select Sub-Agent</option>
                                @foreach($subAgents as $subAgent)
                                    <option value="{{ $subAgent->id }}" 
                                            data-commission="{{ $subAgent->commission }}">
                                        {{ $subAgent->name }} ({{ $subAgent->commission }}%)
                                    </option>
                                @endforeach
                            </select>
                        </div>

                        <!-- Voucher Type -->
                        <div class="col-md-6">
                            <label class="form-label required">Voucher Type</label>
                            <select class="form-select" name="voucher_type" id="voucherTypeSelect" required>
                                <option value="advance">Advance Payment</option>
                                <option value="final">Final Payment</option>
                            </select>
                        </div>

                        <!-- Worker Selection -->
                        <div class="col-12">
                            <div class="card">
                                <div class="card-header bg-light">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <h6 class="mb-0">Select Workers</h6>
                                        <div class="input-group" style="width: 300px;">
                                            <span class="input-group-text bg-light">
                                                <i class="fas fa-search"></i>
                                            </span>
                                            <input type="text" class="form-control" id="workerSearch" 
                                                   placeholder="Search workers...">
                                        </div>
                                    </div>
                                </div>
                                <div class="card-body">
                                    <div class="table-responsive" style="max-height: 300px;">
                                        <table class="table table-hover">
                                            <thead class="sticky-top bg-light">
                                                <tr>
                                                    <th>
                                                        <input type="checkbox" class="form-check-input" id="selectAllWorkers">
                                                    </th>
                                                    <th>Worker Name</th>
                                                    <th>Passport</th>
                                                    <th>Amount</th>
                                                    <th>Action</th>
                                                </tr>
                                            </thead>
                                            <tbody id="workersTableBody">
                                                @foreach($eligibleWorkers as $worker)
                                                    <tr class="worker-row" data-sub-agent-id="{{ $worker->sub_agent_id }}">
                                                        <td>
                                                            <input type="checkbox" class="form-check-input worker-checkbox" 
                                                                   value="{{ $worker->id }}">
                                                        </td>
                                                        <td>{{ $worker->surname }} {{ $worker->given_names }}</td>
                                                        <td>{{ $worker->passport_number }}</td>
                                                        <td class="worker-amount">0.00</td>
                                                        <td>
                                                            <button type="button" class="btn btn-sm btn-primary add-worker" 
                                                                    data-worker-id="{{ $worker->id }}"
                                                                    data-worker-name="{{ $worker->surname }} {{ $worker->given_names }}"
                                                                    data-passport="{{ $worker->passport_number }}"
                                                                    data-sub-agent-id="{{ $worker->sub_agent_id }}"
                                                                    data-sub-agent-name="{{ $worker->subAgent->name }}"
                                                                    data-commission="{{ $worker->subAgent->commission }}">
                                                                <i class="fas fa-plus"></i>
                                                            </button>
                                                        </td>
                                                    </tr>
                                                @endforeach
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Selected Workers Summary -->
                        <div class="col-12">
                            <div class="card">
                                <div class="card-header bg-light">
                                    <h6 class="mb-0">Selected Workers</h6>
                                </div>
                                <div class="card-body">
                                    <div class="table-responsive">
                                        <table class="table table-sm" id="selectedWorkersTable">
                                            <thead>
                                                <tr>
                                                    <th>#</th>
                                                    <th>Worker Name</th>
                                                    <th>Passport</th>
                                                    <th>Amount</th>
                                                    <th>Action</th>
                                                </tr>
                                            </thead>
                                            <tbody></tbody>
                                            <tfoot class="table-light">
                                                <tr>
                                                    <th colspan="3" class="text-end">Total Amount:</th>
                                                    <th id="totalAmount">0.00</th>
                                                    <th></th>
                                                </tr>
                                            </tfoot>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Notes -->
                        <div class="col-12">
                            <label class="form-label">Notes</label>
                            <textarea class="form-control" name="notes" rows="3" 
                                      placeholder="Add any additional notes here..."></textarea>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="submitVoucher()">
                    <i class="fas fa-save me-1"></i>Create Voucher
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Create Invoice Modal -->
<div class="modal fade" id="createInvoiceModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Create New Invoice</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <!-- Available Workers Tab -->
                <div class="tab-pane fade show active" id="workers-pane" role="tabpanel" 
                     aria-labelledby="workers-tab" tabindex="0">
                    <div class="card border-0 shadow-sm">
                        <div class="card-body">
                            <div class="row mb-4">
                                <div class="col-md-6">
                                    <label for="agencyFilter" class="form-label">Filter by Agency</label>
                                    <select id="agencyFilter" class="form-select">
                                        <option value="">All Agencies</option>
                                        @foreach($agencies as $agency)
                                            <option value="{{ $agency->id }}">{{ $agency->company_name }}</option>
                                        @endforeach
                                    </select>
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label fw-bold">Invoice Type:</label>
                                    <div>
                                        <div class="form-check form-check-inline">
                                            <input class="form-check-input" type="checkbox" id="includeStamping" checked>
                                            <label class="form-check-label" for="includeStamping">Stamping</label>
                                        </div>
                                        <div class="form-check form-check-inline">
                                            <input class="form-check-input" type="checkbox" id="includeArrival" checked>
                                            <label class="form-check-label" for="includeArrival">Arrival</label>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="table-responsive mb-4">
                                <table class="table table-hover">
                                    <thead class="table-light">
                                        <tr>
                                            <th>#</th>
                                            <th>Worker Name</th>
                                            <th>Passport Number</th>
                                            <th>Company</th>
                                            <th>Stamping Fee</th>
                                            <th>Arrival Fee</th>
                                            <th>Action</th>
                                        </tr>
                                    </thead>
                                    <tbody id="workersTable">
                                        @foreach($eligibleWorkers as $index => $worker)
                                            @if($worker->contracted_agency)
                                            <tr class="worker-row" data-agency-id="{{ $worker->contracted_agency->id }}">
                                                <td>{{ $index + 1 }}</td>
                                                <td>{{ $worker->surname }} {{ $worker->given_names }}</td>
                                                <td>{{ $worker->passport_number }}</td>
                                                <td>{{ $worker->contracted_agency->company_name }}</td>
                                                <td>{{ number_format($worker->stamping_fee, 2) }}</td>
                                                <td>{{ number_format($worker->arrival_fee, 2) }}</td>
                                                <td>
                                                    <button class="btn btn-sm btn-success add-to-invoice"
                                                        data-worker-id="{{ $worker->id }}"
                                                        data-worker-name="{{ $worker->surname }} {{ $worker->given_names }}"
                                                        data-passport-number="{{ $worker->passport_number }}"
                                                        data-company-id="{{ $worker->contracted_agency->id }}"
                                                        data-company-name="{{ $worker->contracted_agency->company_name }}"
                                                        data-company-email="{{ $worker->contracted_agency->email }}"
                                                        data-company-phone="{{ $worker->contracted_agency->phone }}"
                                                        data-company-city="{{ $worker->contracted_agency->city }}"
                                                        data-company-country="{{ $worker->contracted_agency->country }}"
                                                        data-company-recruitment-fee="{{ $worker->contracted_agency->recruitment_fee }}"
                                                        data-stamping-fee="{{ $worker->stamping_fee }}"
                                                        data-arrival-fee="{{ $worker->arrival_fee }}"
                                                    >
                                                        <i class="fas fa-plus me-1"></i> Add
                                                    </button>
                                                </td>
                                            </tr>
                                            @endif
                                        @endforeach
                                    </tbody>
                                </table>
                            </div>

                            <div class="card mt-4">
                                <div class="card-header bg-light">
                                    <h5 class="mb-0">Create New Invoice</h5>
                                </div>
                                <div class="card-body">
                                    <form action="{{ route('invoices.store') }}" method="POST" id="invoiceForm">
                                        @csrf

                                        <div class="row">
                                            <div class="col-md-6">
                                                <h6 class="fw-bold">Billing Address</h6>
                                                <div class="ps-3" id="billing_address" style="white-space: pre-line; font-family: monospace; font-weight: bold;">
                                                    <!-- Content will be populated by JavaScript -->
                                                </div>

                                                <!-- Hidden Inputs to Submit -->
                                                <input type="hidden" name="client_name" id="client_name">
                                                <input type="hidden" name="client_email" id="client_email">
                                                <input type="hidden" name="client_phone" id="client_phone">
                                                <input type="hidden" name="client_city" id="client_city">
                                                <input type="hidden" name="client_country" id="client_country">
                                                <input type="hidden" name="external_agency_id" id="external_agency_id">
                                                <input type="hidden" name="total_amount" id="total_amount">
                                            </div>
                                            <div class="col-md-6">
                                                <h6 class="fw-bold">Selected Workers</h6>
                                                <div class="table-responsive">
                                                    <table class="table table-sm" id="selectedWorkersTable">
                                                        <thead>
                                                            <tr>
                                                                <th>#</th>
                                                                <th>Name</th>
                                                                <th>Stamping</th>
                                                                <th>Arrival</th>
                                                                <th>Action</th>
                                                                <th></th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <!-- Filled dynamically -->
                                                        </tbody>
                                                        <tfoot class="table-light">
                                                            <tr>
                                                                <th colspan="2" class="text-end">Total</th>
                                                                <th id="totalStampingFee">0.00</th>
                                                                <th id="totalArrivalFee">0.00</th>
                                                                <th></th>
                                                            </tr>
                                                            <tr>
                                                                <th colspan="2" class="text-end">Subtotal</th>
                                                                <th colspan="2" id="subtotal">0.00</th>
                                                                <th></th>
                                                            </tr>
                                                        </tfoot>
                                                    </table>
                                                </div>
                                            </div>
                                        </div>

                                        <div class="d-grid gap-2 d-md-flex justify-content-md-end mt-4">
                                            <button type="submit" class="btn btn-primary">
                                                <i class="fas fa-save me-1"></i> Create Invoice
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // === Invoice Functionality ===
    const selectedWorkerIds = new Set();
    const tableBody = document.querySelector('#selectedWorkersTable tbody');
    const billingAddressDiv = document.getElementById('billing_address');
    const invoiceForm = document.getElementById('invoiceForm');
    const hiddenFieldsContainer = document.createElement('div');
    hiddenFieldsContainer.id = 'hidden-fields-container';
    invoiceForm?.appendChild(hiddenFieldsContainer);

    let currentAgencyId = null;

    // Agency Filter
    const agencyFilter = document.getElementById('agencyFilter');
    agencyFilter?.addEventListener('change', function() {
        const selectedId = this.value;
        document.querySelectorAll('.worker-row').forEach(row => {
            row.style.display = (selectedId === "" || row.dataset.agencyId === selectedId) ? "" : "none";
        });
    });

    // Add Worker to Invoice
    document.querySelectorAll('.add-to-invoice').forEach(button => {
        button.addEventListener('click', function() {
            const workerId = this.dataset.workerId;
            const agencyId = this.dataset.companyId;
            const workerName = this.dataset.workerName;
            const passportNumber = this.dataset.passportNumber;
            const companyName = this.dataset.companyName;
            const companyEmail = this.dataset.companyEmail;
            const companyPhone = this.dataset.companyPhone;
            const companyCity = this.dataset.companyCity;
            const companyCountry = this.dataset.companyCountry;

            // Validate required data
            if (!workerId || !agencyId || !workerName || !passportNumber || !companyName) {
                alert('Required worker or agency information is missing. Please contact support.');
                return;
            }

            const includeStamping = document.getElementById('includeStamping')?.checked ?? true;
            const includeArrival = document.getElementById('includeArrival')?.checked ?? true;

            const stampingFee = includeStamping ? parseFloat(this.dataset.stampingFee) || 0 : 0;
            const arrivalFee = includeArrival ? parseFloat(this.dataset.arrivalFee) || 0 : 0;

            if (selectedWorkerIds.has(workerId)) {
                alert('This worker is already in the invoice.');
                return;
            }

            if (currentAgencyId && currentAgencyId !== agencyId) {
                alert('Only workers from the same agency can be added.');
                return;
            }

            if (selectedWorkerIds.size === 0) {
                currentAgencyId = agencyId;
                document.getElementById('external_agency_id').value = agencyId;
                
                // Set hidden client fields
                document.getElementById('client_name').value = companyName;
                document.getElementById('client_email').value = companyEmail || '';
                document.getElementById('client_phone').value = companyPhone || '';
                document.getElementById('client_city').value = companyCity || '';
                document.getElementById('client_country').value = companyCountry || '';

                if (billingAddressDiv) {
                    billingAddressDiv.innerHTML = `
                        ${companyName}
                        ${companyEmail || 'No email provided'}
                        Phone: ${companyPhone || 'No phone provided'}
                        ${companyCity || 'No city'}, ${companyCountry || 'No country'}
                    `;
                }
            }

            // Create worker data object
            const workerData = {
                id: workerId,
                name: workerName,
                passport: passportNumber,
                stamping_fee: stampingFee,
                arrival_fee: arrivalFee
            };

            const row = document.createElement('tr');
            row.className = 'selected-worker-row';
            row.dataset.workerId = workerId;
            row.dataset.stampingFee = stampingFee;
            row.dataset.arrivalFee = arrivalFee;
            row.innerHTML = `
                <td>${tableBody.children.length + 1}</td>
                <td>${workerName}</td>
                <td>${stampingFee.toFixed(2)}</td>
                <td>${arrivalFee.toFixed(2)}</td>
                <td class="text-end">
                    <button type="button" class="btn btn-sm btn-outline-danger remove-worker-btn" data-worker-id="${workerId}">
                        <i class="fas fa-times"></i>
                    </button>
                </td>
                <td></td>
            `;
            tableBody.appendChild(row);

            // Add hidden fields for form submission
            hiddenFieldsContainer.innerHTML += `
                <input type="hidden" name="workers[]" value="${workerId}">
                <input type="hidden" name="worker_names[]" value="${workerName}">
                <input type="hidden" name="passport_numbers[]" value="${passportNumber}">
                <input type="hidden" name="stamping_fees[]" value="${stampingFee}">
                <input type="hidden" name="arrival_fees[]" value="${arrivalFee}">
            `;

            selectedWorkerIds.add(workerId);
            this.disabled = true;
            this.classList.replace('btn-success', 'btn-secondary');

            calculateInvoice();
        });
    });

    // Remove Worker from Invoice
    tableBody?.addEventListener('click', function(e) {
        if (e.target.closest('.remove-worker-btn')) {
            const btn = e.target.closest('.remove-worker-btn');
            const workerId = btn.dataset.workerId;
            selectedWorkerIds.delete(workerId);
            btn.closest('tr')?.remove();

            const addButton = document.querySelector(`.add-to-invoice[data-worker-id="${workerId}"]`);
            if (addButton) {
                addButton.disabled = false;
                addButton.classList.replace('btn-secondary', 'btn-success');
            }

            if (selectedWorkerIds.size === 0) {
                currentAgencyId = null;
                // Clear hidden client fields
                document.getElementById('client_name').value = '';
                document.getElementById('client_email').value = '';
                document.getElementById('client_phone').value = '';
                document.getElementById('client_city').value = '';
                document.getElementById('client_country').value = '';
                if (billingAddressDiv) billingAddressDiv.innerHTML = '';
            } else {
                document.querySelectorAll('#selectedWorkersTable tbody tr').forEach((row, idx) => {
                    row.cells[0].textContent = idx + 1;
                });
            }

            hiddenFieldsContainer.querySelectorAll(`input[value="${workerId}"]`).forEach(input => input.remove());
            calculateInvoice();
        }
    });

    // Calculate Invoice Totals
    window.calculateInvoice = function() {
        let totalStamping = 0, totalArrival = 0;
        document.querySelectorAll('#selectedWorkersTable tbody tr').forEach(row => {
            totalStamping += parseFloat(row.dataset.stampingFee) || 0;
            totalArrival += parseFloat(row.dataset.arrivalFee) || 0;
        });

        const subtotal = totalStamping + totalArrival;
        document.getElementById('totalStampingFee').textContent = totalStamping.toFixed(2);
        document.getElementById('totalArrivalFee').textContent = totalArrival.toFixed(2);
        document.getElementById('subtotal').textContent = subtotal.toFixed(2);
        document.getElementById('total_amount').value = subtotal.toFixed(2);
    };

    // Form Validation
    invoiceForm?.addEventListener('submit', function(e) {
        e.preventDefault();
        
        if (selectedWorkerIds.size === 0) {
            alert('Please add at least one worker to the invoice.');
            return;
        }

        // Validate required fields
        const requiredFields = ['client_name', 'client_email', 'client_phone', 'client_city', 'client_country'];
        for (const field of requiredFields) {
            const value = document.getElementById(field).value.trim();
            if (!value) {
                alert(`Please ensure all client information is filled out.`);
                return;
            }
        }

        // Validate total amount
        const totalAmount = parseFloat(document.getElementById('total_amount').value);
        if (isNaN(totalAmount) || totalAmount <= 0) {
            alert('Please ensure the total amount is valid.');
            return;
        }

        // If all validations pass, submit the form
        this.submit();
    });

    // Reset form when modal is closed
    document.getElementById('createInvoiceModal').addEventListener('hidden.bs.modal', function () {
        document.getElementById('invoiceForm').reset();
        document.getElementById('billing_address').innerHTML = '';
        document.querySelector('#selectedWorkersTable tbody').innerHTML = '';
        selectedWorkerIds.clear();
        currentAgencyId = null;
        
        // Reset all add buttons
        document.querySelectorAll('.add-to-invoice').forEach(button => {
            button.disabled = false;
            button.classList.replace('btn-secondary', 'btn-success');
        });

        // Clear hidden fields
        document.getElementById('client_name').value = '';
        document.getElementById('client_email').value = '';
        document.getElementById('client_phone').value = '';
        document.getElementById('client_city').value = '';
        document.getElementById('client_country').value = '';
        document.getElementById('external_agency_id').value = '';
        document.getElementById('total_amount').value = '';
        
        calculateInvoice();
    });

    // Handle fee type changes
    document.getElementById('includeStamping')?.addEventListener('change', function() {
        recalculateFees();
    });

    document.getElementById('includeArrival')?.addEventListener('change', function() {
        recalculateFees();
    });

    // Recalculate fees when fee types change
    function recalculateFees() {
        const includeStamping = document.getElementById('includeStamping')?.checked ?? true;
        const includeArrival = document.getElementById('includeArrival')?.checked ?? true;

        document.querySelectorAll('.selected-worker-row').forEach(row => {
            const workerId = row.dataset.workerId;
            const addButton = document.querySelector(`.add-to-invoice[data-worker-id="${workerId}"]`);
            
            if (addButton) {
                const stampingFee = includeStamping ? parseFloat(addButton.dataset.stampingFee) || 0 : 0;
                const arrivalFee = includeArrival ? parseFloat(addButton.dataset.arrivalFee) || 0 : 0;
                
                row.dataset.stampingFee = stampingFee;
                row.dataset.arrivalFee = arrivalFee;
                
                row.cells[2].textContent = stampingFee.toFixed(2);
                row.cells[3].textContent = arrivalFee.toFixed(2);
            }
        });

        calculateInvoice();
    }
});

 // General function to show SweetAlert
 function showAlert(icon, title, text, confirmButtonText = 'OK') {
            Swal.fire({
                icon: icon,
                title: title,
                text: text,
                showConfirmButton: false,
                        timer: 3000,
                        timerProgressBar: true,
                        width: 'auto',
                        padding: '0.5em',
                        customClass: {
                            popup: 'colored-toast'
                        }
            });
        }

        // Success alert
        function showSuccessAlert(title, text) {
            showAlert('success', title, text);
        }

        // Error alert
        function showErrorAlert(title, text) {
            showAlert('error', title, text);
        }

        // Confirmation dialog
        function showConfirmDialog(title, text, confirmButtonText, callback) {
            Swal.fire({
                title: title,
                text: text,
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#3085d6',
                cancelButtonColor: '#d33',
                confirmButtonText: confirmButtonText
            }).then((result) => {
                if (result.isConfirmed) {
                    callback();
                }
            });
        }

        // Mark as Paid confirmation
        document.addEventListener('DOMContentLoaded', function() {
            // Handle mark as paid buttons
            document.querySelectorAll('form[action*="markAsPaid"]').forEach(form => {
                form.addEventListener('submit', function(e) {
                    e.preventDefault();
                    
                    showConfirmDialog(
                        'Mark Invoice as Paid?',
                        'Are you sure you want to mark this invoice as paid? This action cannot be undone.',
                        'Yes, mark as paid',
                        () => form.submit()
                    );
                });
            });

            // Handle invoice creation success/error
            @if(session('success'))
                showSuccessAlert('Success!', '{{ session('success') }}');
            @endif

            @if(session('error'))
                showErrorAlert('Error!', '{{ session('error') }}');
            @endif

            // Handle voucher actions
            window.approveVoucher = function(voucherId) {
                showConfirmDialog(
                    'Approve Voucher?',
                    'Are you sure you want to approve this payment voucher?',
                    'Yes, approve',
                    () => {
                        fetch(`/vouchers/${voucherId}/approve`, {
                            method: 'POST',
                            headers: {
                                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                                'Content-Type': 'application/json',
                                'Accept': 'application/json'
                            }
                        })
                        .then(response => response.json())
                        .then(data => {
                            if (data.success) {
                                showSuccessAlert('Approved!', 'The voucher has been approved successfully.');
                                setTimeout(() => window.location.reload(), 1500);
                            } else {
                                showErrorAlert('Error!', data.message || 'Failed to approve voucher.');
                            }
                        })
                        .catch(error => {
                            showErrorAlert('Error!', 'An error occurred while approving the voucher.');
                        });
                    }
                );
            };

            window.rejectVoucher = function(voucherId) {
                Swal.fire({
                    title: 'Reject Voucher?',
                    text: 'Please provide a reason for rejecting this voucher:',
                    icon: 'warning',
                    input: 'text',
                    inputPlaceholder: 'Reason for rejection...',
                    showCancelButton: true,
                    confirmButtonColor: '#d33',
                    cancelButtonColor: '#3085d6',
                    confirmButtonText: 'Reject',
                    preConfirm: (reason) => {
                        if (!reason) {
                            Swal.showValidationMessage('Please provide a reason');
                        }
                        return reason;
                    }
                }).then((result) => {
                    if (result.isConfirmed) {
                        fetch(`/vouchers/${voucherId}/reject`, {
                            method: 'POST',
                            headers: {
                                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                                'Content-Type': 'application/json',
                                'Accept': 'application/json'
                            },
                            body: JSON.stringify({ reason: result.value })
                        })
                        .then(response => response.json())
                        .then(data => {
                            if (data.success) {
                                showSuccessAlert('Rejected!', 'The voucher has been rejected successfully.');
                                setTimeout(() => window.location.reload(), 1500);
                            } else {
                                showErrorAlert('Error!', data.message || 'Failed to reject voucher.');
                            }
                        })
                        .catch(error => {
                            showErrorAlert('Error!', 'An error occurred while rejecting the voucher.');
                        });
                    }
                });
            };

            // Enhanced form submission with SweetAlert
            document.getElementById('invoiceForm')?.addEventListener('submit', function(e) {
                e.preventDefault();
                
                if (selectedWorkerIds.size === 0) {
                    showErrorAlert('Error!', 'Please add at least one worker to the invoice.');
                    return;
                }

                // Show loading indicator
                Swal.fire({
                    title: 'Creating Invoice',
                    html: 'Please wait while we process your invoice...',
                    allowOutsideClick: false,
                    didOpen: () => {
                        Swal.showLoading();
                    }
                });

                // Submit the form
                this.submit();
            });

            // Voucher form submission
            document.getElementById('voucherForm')?.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const selectedWorkers = document.querySelectorAll('#selectedWorkersTable tbody tr').length;
                if (selectedWorkers === 0) {
                    showErrorAlert('Error!', 'Please select at least one worker for the voucher.');
                    return;
                }

                // Show loading indicator
                Swal.fire({
                    title: 'Creating Voucher',
                    html: 'Please wait while we process your voucher...',
                    allowOutsideClick: false,
                    didOpen: () => {
                        Swal.showLoading();
                    }
                });

                // Submit the form
                this.submit();
            });
        });

</script>




<style>
.nav-tabs .nav-link {
    font-weight: 500;
    padding: 0.75rem 1.5rem;
    transition: all 0.3s ease;
    border-radius: 0.5rem 0.5rem 0 0;
    margin-right: 0.5rem;
    border: 1px solid transparent;
}

.nav-tabs .nav-link.active {
    background-color: #fff;
    border-color: #dee2e6 #dee2e6 #fff;
    color: #0d6efd;
    border-bottom: 3px solid #0d6efd;
}

.tab-content {
    padding: 1.5rem 0;
    border-left: 1px solid #dee2e6;
    border-right: 1px solid #dee2e6;
    border-bottom: 1px solid #dee2e6;
    border-radius: 0 0 0.5rem 0.5rem;
    background-color: #fff;
}

.worker-row:hover {
    background-color: rgba(13, 110, 253, 0.05);
}

.selected-worker-row {
    background-color: rgba(25, 135, 84, 0.05);
}

#billing_address {
    min-height: 120px;
    border: 1px solid #dee2e6;
    padding: 1rem;
    border-radius: 0.25rem;
}

#selectedWorkersTable tbody tr:hover {
    background-color: rgba(0, 0, 0, 0.02);
}

.add-to-invoice {
    transition: all 0.2s ease;
}

.add-to-invoice:hover {
    transform: translateY(-1px);
}

.remove-worker-btn {
    transition: all 0.2s ease;
}

.remove-worker-btn:hover {
    transform: scale(1.1);
}

/* Voucher specific styles */
.avatar-sm {
    width: 32px;
    height: 32px;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    font-size: 0.875rem;
    color: #fff;
}

.required:after {
    content: " *";
    color: red;
}

.sticky-top {
    position: sticky;
    top: 0;
    z-index: 1;
}

.table-responsive {
    scrollbar-width: thin;
    scrollbar-color: #dee2e6 #fff;
}

.table-responsive::-webkit-scrollbar {
    width: 6px;
    height: 6px;
}

.table-responsive::-webkit-scrollbar-track {
    background: #fff;
}

.table-responsive::-webkit-scrollbar-thumb {
    background-color: #dee2e6;
    border-radius: 3px;
}

.modal-lg {
    max-width: 900px;
}

.form-check-input:checked {
    background-color: #0d6efd;
    border-color: #0d6efd;
}

.btn-group .btn {
    padding: 0.25rem 0.5rem;
}

.btn-group .btn i {
    font-size: 0.875rem;
}

.badge {
    padding: 0.5em 0.75em;
    font-weight: 500;
}

/* Animation for adding/removing workers */
@keyframes fadeIn {
    from { opacity: 0; transform: translateY(-10px); }
    to { opacity: 1; transform: translateY(0); }
}

.selected-worker-row {
    animation: fadeIn 0.3s ease-out;
}

/* Additional styles for better UX */
.selected-worker-row {
    transition: all 0.3s ease;
}

.selected-worker-row:hover {
    background-color: rgba(25, 135, 84, 0.1);
}

.add-to-invoice:disabled {
    cursor: not-allowed;
    opacity: 0.7;
}

#billing_address {
    background-color: #f8f9fa;
    border: 1px solid #dee2e6;
    border-radius: 0.25rem;
    padding: 1rem;
    margin-bottom: 1rem;
}

.form-check-input:checked {
    background-color: #198754;
    border-color: #198754;
}

.btn-outline-danger:hover {
    background-color: #dc3545;
    color: white;
}

/* Animation for adding/removing workers */
@keyframes fadeIn {
    from { opacity: 0; transform: translateY(-10px); }
    to { opacity: 1; transform: translateY(0); }
}

.selected-worker-row {
    animation: fadeIn 0.3s ease-out;
}

/* Responsive adjustments */
@media (max-width: 768px) {
    .modal-lg {
        max-width: 95%;
        margin: 1.75rem auto;
    }
    
    .table-responsive {
        margin-bottom: 1rem;
    }
}
</style>
@endsection